<?php
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/auth.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

$user = authenticateRequest();
if (!$user) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit();
}

$db = Database::getInstance();
$method = $_SERVER['REQUEST_METHOD'];
$path = $_SERVER['PATH_INFO'] ?? '/';

try {
    // GET /api/targets/current - Get current target with progress
    if ($method === 'GET' && $path === '/current') {
        $target_type = $_GET['target_type'] ?? 'monthly';
        $user_id = $_GET['user_id'] ?? $user['id'];
        
        // Only admins can view other users' targets
        if ($user_id != $user['id'] && $user['role'] !== 'admin') {
            http_response_code(403);
            echo json_encode(['error' => 'Only admins can view other users\' targets']);
            exit();
        }
        
        $today = date('Y-m-d H:i:s');
        $startDate = '';
        $endDate = '';
        
        switch ($target_type) {
            case 'daily':
                $startDate = date('Y-m-d 00:00:00');
                $endDate = date('Y-m-d 23:59:59');
                break;
            case 'weekly':
                $dayOfWeek = date('w');
                $startDate = date('Y-m-d 00:00:00', strtotime("-$dayOfWeek days"));
                $endDate = date('Y-m-d 23:59:59', strtotime("-$dayOfWeek days +6 days"));
                break;
            case 'monthly':
                $startDate = date('Y-m-01 00:00:00');
                $endDate = date('Y-m-t 23:59:59');
                break;
            case 'yearly':
                $startDate = date('Y-01-01 00:00:00');
                $endDate = date('Y-12-31 23:59:59');
                break;
        }
        
        // Get current target
        $target = $db->fetchOne(
            "SELECT * FROM sales_targets 
             WHERE user_id = ? AND target_type = ? 
             AND period_start <= ? AND period_end >= ?
             ORDER BY created_at DESC
             LIMIT 1",
            [$user_id, $target_type, $startDate, $endDate]
        );
        
        // Get current progress - use amount_paid for partial payments
        $progress = $db->fetchOne(
            "SELECT 
                COUNT(*) as sales_count,
                COALESCE(SUM(amount_paid), 0) as total_amount,
                COALESCE(SUM(total_profit), 0) as total_profit
             FROM sales 
             WHERE user_id = ? AND created_at >= ? AND created_at <= ?",
            [$user_id, $startDate, $endDate]
        );
        
        echo json_encode([
            'data' => [
                'target' => $target,
                'progress' => $progress,
                'period' => [
                    'start' => $startDate,
                    'end' => $endDate,
                    'type' => $target_type
                ]
            ]
        ]);
        exit();
    }
    
    // GET /api/targets - Get all targets
    if ($method === 'GET') {
        $target_type = $_GET['target_type'] ?? 'monthly';
        $user_id = $_GET['user_id'] ?? $user['id'];
        
        // Only admins can view other users' targets
        if ($user_id != $user['id'] && $user['role'] !== 'admin') {
            http_response_code(403);
            echo json_encode(['error' => 'Only admins can view other users\' targets']);
            exit();
        }
        
        $targets = $db->fetchAll(
            "SELECT * FROM sales_targets 
             WHERE user_id = ? AND target_type = ?
             ORDER BY period_start DESC
             LIMIT 10",
            [$user_id, $target_type]
        );
        
        echo json_encode(['targets' => $targets]);
        exit();
    }
    
    // POST /api/targets - Create or update target
    if ($method === 'POST') {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $target_user_id = $data['user_id'] ?? $user['id'];
        $target_type = $data['target_type'];
        $target_amount = $data['target_amount'];
        $target_sales_count = $data['target_sales_count'];
        $period_start = $data['period_start'];
        $period_end = $data['period_end'];
        
        // Only admins can set targets for other users
        if ($target_user_id != $user['id'] && $user['role'] !== 'admin') {
            http_response_code(403);
            echo json_encode(['error' => 'Only admins can set targets for other users']);
            exit();
        }
        
        // Check if target already exists
        $existing = $db->fetchOne(
            "SELECT id FROM sales_targets 
             WHERE user_id = ? AND target_type = ? 
             AND period_start = ? AND period_end = ?",
            [$target_user_id, $target_type, $period_start, $period_end]
        );
        
        if ($existing) {
            // Update existing target
            $db->execute(
                "UPDATE sales_targets 
                 SET target_amount = ?, target_sales_count = ?, updated_at = CURRENT_TIMESTAMP
                 WHERE id = ?",
                [$target_amount, $target_sales_count, $existing['id']]
            );
            
            echo json_encode(['message' => 'Target updated successfully', 'id' => $existing['id']]);
        } else {
            // Create new target - set default daily target to 1000
            if ($target_type === 'daily' && empty($target_amount)) {
                $target_amount = 1000;
            }
            
            $result = $db->execute(
                "INSERT INTO sales_targets (user_id, target_type, target_amount, target_sales_count, period_start, period_end)
                 VALUES (?, ?, ?, ?, ?, ?)",
                [$target_user_id, $target_type, $target_amount, $target_sales_count, $period_start, $period_end]
            );
            
            echo json_encode(['message' => 'Target created successfully', 'id' => $result['lastInsertId']]);
        }
        exit();
    }
    
    // DELETE /api/targets/:id - Delete target
    if ($method === 'DELETE' && preg_match('/^\/(\d+)$/', $path, $matches)) {
        $id = $matches[1];
        
        $db->execute(
            "DELETE FROM sales_targets 
             WHERE id = ? AND user_id = ?",
            [$id, $user['id']]
        );
        
        echo json_encode(['message' => 'Target deleted successfully']);
        exit();
    }
    
    http_response_code(404);
    echo json_encode(['error' => 'Not found']);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
