<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/helpers.php';

handleCORS();

$method = $_SERVER['REQUEST_METHOD'];
$db = Database::getInstance();

switch ($method) {
    case 'GET':
        $user = requireAuth();
        
        if (isset($_GET['id'])) {
            // Get single product
            $product = $db->fetchOne(
                'SELECT p.*, c.name as category_name, s.name as supplier_name FROM products p LEFT JOIN categories c ON p.category_id = c.id LEFT JOIN suppliers s ON p.supplier_id = s.id WHERE (p.id = ? OR p.barcode = ? OR p.product_id = ?) AND p.is_active = TRUE',
                [$_GET['id'], $_GET['id'], $_GET['id']]
            );
            
            if (!$product) {
                errorResponse('Product not found', 404);
            }
            
            successResponse(['product' => $product]);
            
        } elseif (isset($_GET['low-stock']) || (isset($_GET['alerts']) && $_GET['alerts'] === 'low-stock')) {
            // Get low stock products - handle both /products/low-stock and /products/alerts/low-stock
            $products = $db->fetchAll(
                'SELECT p.*, c.name as category_name FROM products p LEFT JOIN categories c ON p.category_id = c.id WHERE p.quantity_in_stock <= p.low_stock_threshold AND p.is_active = TRUE ORDER BY p.quantity_in_stock ASC'
            );
            
            successResponse(['products' => $products]);
            
        } else {
            // Get all products with filters
            $page = intval($_GET['page'] ?? 1);
            $limit = intval($_GET['limit'] ?? 20);
            $offset = ($page - 1) * $limit;
            
            $conditions = ['p.is_active = TRUE'];
            $params = [];
            
            if (!empty($_GET['search'])) {
                $conditions[] = '(p.name LIKE ? OR p.brand LIKE ? OR p.barcode LIKE ?)';
                $searchTerm = '%' . $_GET['search'] . '%';
                $params[] = $searchTerm;
                $params[] = $searchTerm;
                $params[] = $searchTerm;
            }
            
            if (!empty($_GET['category_id'])) {
                $conditions[] = 'p.category_id = ?';
                $params[] = $_GET['category_id'];
            }
            
            if (!empty($_GET['brand'])) {
                $conditions[] = 'p.brand = ?';
                $params[] = $_GET['brand'];
            }
            
            if (isset($_GET['low_stock']) && $_GET['low_stock'] === 'true') {
                $conditions[] = 'p.quantity_in_stock <= p.low_stock_threshold';
            }
            
            $where = 'WHERE ' . implode(' AND ', $conditions);
            $queryParams = array_merge($params, [$limit, $offset]);
            
            $products = $db->fetchAll(
                "SELECT p.*, c.name as category_name, s.name as supplier_name FROM products p LEFT JOIN categories c ON p.category_id = c.id LEFT JOIN suppliers s ON p.supplier_id = s.id $where ORDER BY p.created_at DESC LIMIT ? OFFSET ?",
                $queryParams
            );
            
            $total = $db->fetchOne("SELECT COUNT(*) as total FROM products p $where", $params)['total'];
            
            successResponse([
                'products' => $products,
                'pagination' => [
                    'page' => $page,
                    'limit' => $limit,
                    'total' => intval($total),
                    'pages' => ceil($total / $limit)
                ]
            ]);
        }
        break;
        
    case 'POST':
        $user = requireAuth();
        requireRole(['admin'], $user);
        
        // Handle both JSON and form data
        $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
        if (strpos($contentType, 'multipart/form-data') !== false) {
            // Handle form data (with file uploads)
            $data = $_POST;
            // Handle file upload if present
            if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
                // File upload logic would go here
                // For now, we'll skip file handling
            }
        } else {
            // Handle JSON data
            $data = getRequestBody();
        }
        
        validateRequired($data, ['name', 'brand', 'category_id', 'size_ml', 'cost_price', 'selling_price']);
        
        // Validate size_unit
        $size_unit = $data['size_unit'] ?? 'ml';
        if (!in_array($size_unit, ['ml', 'g'])) {
            errorResponse("Invalid size_unit. Must be 'ml' or 'g'", 400);
        }
        
        // Generate barcode if not provided
        $barcode = $data['barcode'] ?? ('CHI' . str_pad(mt_rand(1, 999999999), 9, '0', STR_PAD_LEFT));
        
        // Check if barcode exists
        $existing = $db->fetchOne('SELECT id FROM products WHERE barcode = ?', [$barcode]);
        if ($existing) {
            errorResponse('Barcode already exists', 400);
        }
        
        // Generate product ID
        $product_id = 'CHI-' . substr(time(), -6);
        
        $result = $db->execute(
            'INSERT INTO products (product_id, name, brand, category_id, supplier_id, size_ml, size_unit, cost_price, selling_price, quantity_in_stock, low_stock_threshold, barcode, description) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
            [
                $product_id,
                $data['name'],
                $data['brand'],
                $data['category_id'],
                $data['supplier_id'] ?? null,
                $data['size_ml'],
                $size_unit,
                $data['cost_price'],
                $data['selling_price'],
                $data['quantity_in_stock'] ?? 0,
                $data['low_stock_threshold'] ?? 10,
                $barcode,
                $data['description'] ?? null
            ]
        );
        
        successResponse([
            'productId' => $result['lastInsertId'],
            'product_id' => $product_id,
            'barcode' => $barcode
        ], 'Product created successfully', 201);
        break;
        
    case 'PUT':
        $user = requireAuth();
        requireRole(['admin'], $user);
        
        if (!isset($_GET['id'])) {
            errorResponse('Product ID required', 400);
        }
        
        $data = getRequestBody();
        
        // Validate size_unit if provided
        if (isset($data['size_unit']) && !in_array($data['size_unit'], ['ml', 'g'])) {
            errorResponse("Invalid size_unit. Must be 'ml' or 'g'", 400);
        }
        
        $updateFields = [];
        $updateValues = [];
        
        $allowedFields = ['name', 'brand', 'category_id', 'supplier_id', 'size_ml', 'size_unit', 'cost_price', 'selling_price', 'quantity_in_stock', 'low_stock_threshold', 'barcode', 'description'];
        
        foreach ($allowedFields as $field) {
            if (array_key_exists($field, $data)) {
                $updateFields[] = "$field = ?";
                $updateValues[] = $data[$field];
            }
        }
        
        if (empty($updateFields)) {
            errorResponse('No fields to update', 400);
        }
        
        $updateValues[] = $_GET['id'];
        
        $db->execute(
            'UPDATE products SET ' . implode(', ', $updateFields) . ' WHERE id = ?',
            $updateValues
        );
        
        successResponse([], 'Product updated successfully');
        break;
        
    case 'DELETE':
        $user = requireAuth();
        requireRole(['admin'], $user);
        
        if (!isset($_GET['id'])) {
            errorResponse('Product ID required', 400);
        }
        
        $db->execute('UPDATE products SET is_active = FALSE WHERE id = ?', [$_GET['id']]);
        
        successResponse([], 'Product deleted successfully');
        break;
}

errorResponse('Invalid request', 400);
?>